--[[
Copyright (C) GtX (Andy), 2020

Author: GtX | Andy
Date: 24.08.2020
Revision: FS25-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
]]

VehicleSpeedSync = {}
VehicleSpeedSync.DEFAULT_RANGE = 25

VehicleSpeedSync.SPEED_SYNC_VISIBLE = true
VehicleSpeedSync.SPEED_SYNC_ACTIVE = true

VehicleSpeedSync.CRUISE_SYNC_VISIBLE = true
VehicleSpeedSync.CRUISE_SYNC_ACTIVE = true

VehicleSpeedSync.HUD_ICON_VISIBLE = true

VehicleSpeedSync.MOD_NAME = g_currentModName
VehicleSpeedSync.MOD_DIR = g_currentModDirectory

VehicleSpeedSync.SPEC_NAME = string.format("%s.vehicleSpeedSync", VehicleSpeedSync.MOD_NAME)
VehicleSpeedSync.SPEC_TABLE_NAME = string.format("spec_%s", VehicleSpeedSync.SPEC_NAME)

function VehicleSpeedSync.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Drivable, specializations) and not SpecializationUtil.hasSpecialization(Locomotive, specializations)
end

function VehicleSpeedSync.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncFindVehicleInRange",    VehicleSpeedSync.vehicleSpeedSyncFindVehicleInRange)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncGetVehicleIsInRange",   VehicleSpeedSync.vehicleSpeedSyncGetVehicleIsInRange)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncGetCanSyncSpeed",       VehicleSpeedSync.vehicleSpeedSyncGetCanSyncSpeed)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncGetSyncedVehicleSpeed", VehicleSpeedSync.vehicleSpeedSyncGetSyncedVehicleSpeed)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncSetSyncedSpeedState",   VehicleSpeedSync.vehicleSpeedSyncSetSyncedSpeedState)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncGetSyncedSpeedActive",  VehicleSpeedSync.vehicleSpeedSyncGetSyncedSpeedActive)
    SpecializationUtil.registerFunction(vehicleType, "vehicleSpeedSyncGetSyncedSpeedWaiting", VehicleSpeedSync.vehicleSpeedSyncGetSyncedSpeedWaiting)
end

function VehicleSpeedSync.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "updateVehiclePhysics",  VehicleSpeedSync.updateVehiclePhysics)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanStartAIVehicle",  VehicleSpeedSync.getCanStartAIVehicle)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "setCruiseControlState", VehicleSpeedSync.setCruiseControlState)
end

function VehicleSpeedSync.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad",                 VehicleSpeedSync)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete",               VehicleSpeedSync)
    SpecializationUtil.registerEventListener(vehicleType, "onDraw",                 VehicleSpeedSync)
    SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle",         VehicleSpeedSync)
    SpecializationUtil.registerEventListener(vehicleType, "onEnterVehicle",         VehicleSpeedSync)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", VehicleSpeedSync)
end

function VehicleSpeedSync:onLoad(savegame)
    self.spec_vehicleSpeedSync = self[VehicleSpeedSync.SPEC_TABLE_NAME]

    local spec = self.spec_vehicleSpeedSync

    spec.setRange = VehicleSpeedSync.DEFAULT_RANGE

    spec.vehicleInRange = nil
    spec.speedSyncActive = false
    spec.isWaiting = false

    spec.noVehicleInRangeText = string.format("%s\n%s", g_i18n:getText("warning_actionNotAllowedNow"), g_i18n:getText("shop_noVehicles"))

    if self.isClient then
        spec.vehicleName = self:getName()
        spec.speedSyncText = g_i18n:getText("state_speedSync", VehicleSpeedSync.MOD_NAME)

        local xmlFile = loadXMLFile("SpeedSyncSounds", Utils.getFilename("sounds/speedSyncSounds.xml", VehicleSpeedSync.MOD_DIR))

        if xmlFile ~= nil then
            spec.samples = {
                activate = g_soundManager:loadSampleFromXML(xmlFile, "speedSyncSounds", "activate", VehicleSpeedSync.MOD_DIR, self.components, 1, AudioGroup.VEHICLE, nil, self),
                deactivate = g_soundManager:loadSampleFromXML(xmlFile, "speedSyncSounds", "deactivate", VehicleSpeedSync.MOD_DIR, self.components, 1, AudioGroup.VEHICLE, nil, self),
                connectionLost = g_soundManager:loadSampleFromXML(xmlFile, "speedSyncSounds", "connectionLost", VehicleSpeedSync.MOD_DIR, self.components, 1, AudioGroup.VEHICLE, nil, self)
            }

            delete(xmlFile)
        else
            spec.samples = {}
        end
    end
end

function VehicleSpeedSync:onDelete()
    local spec = self.spec_vehicleSpeedSync

    self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, false, true)

    if self.isClient ~= nil and spec ~= nil then
        for _, sample in pairs(spec.samples) do
            g_soundManager:deleteSample(sample)
        end
    end
end

function VehicleSpeedSync:onDraw(isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if self.isClient and self.getIsEntered ~= nil and self:getIsEntered() and self:vehicleSpeedSyncGetSyncedSpeedActive() then
        local spec = self.spec_vehicleSpeedSync
        local vehicleName, controllerName = nil, "AI"

        if not spec.vehicleInRange:getIsAIActive() then
            if spec.vehicleInRange.getControllerName ~= nil then
                controllerName = spec.vehicleInRange:getControllerName()
            else
                controllerName = "Player"
            end
        end

        if spec.vehicleInRange.spec_vehicleSpeedSync ~= nil then
            vehicleName = spec.vehicleInRange.spec_vehicleSpeedSync.vehicleName
        end

        if vehicleName == nil then
            vehicleName = spec.vehicleInRange:getName()
        end

        g_currentMission:addExtraPrintText(string.format(spec.speedSyncText, vehicleName, controllerName))

        -- Stop the HUD flashing if vehicle starts to move
        if spec.isWaiting and spec.vehicleInRange ~= nil then
            if spec.vehicleInRange:getLastSpeed() > 1 then
                spec.isWaiting = false
            end
        end
    end
end

function VehicleSpeedSync:updateDebugValues(values)
    local spec = self.spec_vehicleSpeedSync

    if spec ~= nil then
        local vehicle = spec.vehicleInRange

        if vehicle == nil or g_currentMission.vehicleSystem.interactiveVehicles[vehicle] == nil then
            vehicle = self:vehicleSpeedSyncFindVehicleInRange(spec.setRange)
        end

        if vehicle ~= nil then
            local speed = math.max(0, vehicle:getLastSpeed() * vehicle.spec_motorized.speedDisplayScale)

            if speed < 0.5 then
                speed = 0
            end

            table.insert(values, {
                name = "Vehicle Speed:",
                value = string.format("%.1f", speed)
            })

            table.insert(values, {
                name = "Waiting:",
                value = tostring(spec.isWaiting)
            })

            table.insert(values, {
                name = "Vehicle Distance:",
                value = string.format("%.2f", calcDistanceFrom(self.rootNode, vehicle.rootNode) or 0)
            })
        end
    end
end

function VehicleSpeedSync:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self.spec_vehicleSpeedSync

        self:clearActionEventsTable(spec.actionEvents)
        spec.toggleSpeedSyncEventId = nil

        if self:getIsActiveForInput(true, false) and self.getIsEntered ~= nil and self:getIsEntered() then
            if Utils.getNoNil(VehicleSpeedSync.CRUISE_SYNC_ACTIVE, true) then
                local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.SYNC_CRUISE_CONTROL, self, VehicleSpeedSync.actionEventSyncCruiseControl, false, true, false, true)
                g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
                g_inputBinding:setActionEventTextVisibility(actionEventId, VehicleSpeedSync.CRUISE_SYNC_VISIBLE)
            end

            if Utils.getNoNil(VehicleSpeedSync.SPEED_SYNC_ACTIVE, true) then
                local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_SPEED_SYNC, self, VehicleSpeedSync.actionEventSpeedSyncState, false, true, false, true)
                g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("action_activateSpeedSync", VehicleSpeedSync.MOD_NAME))
                g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
                g_inputBinding:setActionEventTextVisibility(actionEventId, VehicleSpeedSync.SPEED_SYNC_VISIBLE)

                spec.toggleSpeedSyncEventId = actionEventId
            end
        end
    end
end

function VehicleSpeedSync:onLeaveVehicle()
    self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, false, true)
end

function VehicleSpeedSync:onEnterVehicle(isControlling)
    self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, false, true)
end

function VehicleSpeedSync:vehicleSpeedSyncFindVehicleInRange(range)
    local vehicleInRange

    if self.isClient and self:vehicleSpeedSyncGetCanSyncSpeed() then
        local lastDistance = math.huge

        for _, vehicle in pairs (g_currentMission.vehicleSystem.interactiveVehicles) do
            if vehicle ~= self and vehicle.spec_drivable ~= nil and vehicle:getIsMotorStarted() then
                local distance = calcDistanceFrom(self.rootNode, vehicle.rootNode)

                if distance <= (range or VehicleSpeedSync.DEFAULT_RANGE) and distance < lastDistance then
                    lastDistance = distance
                    vehicleInRange = vehicle
                end
            end
        end
    end

    return vehicleInRange
end

function VehicleSpeedSync:vehicleSpeedSyncGetVehicleIsInRange(vehicle, range)
    if vehicle ~= nil and g_currentMission.vehicleSystem.interactiveVehicles[vehicle] ~= nil then
        return calcDistanceFrom(self.rootNode, vehicle.rootNode) <= (range or VehicleSpeedSync.DEFAULT_RANGE)
    end

    return false
end

function VehicleSpeedSync:vehicleSpeedSyncGetCanSyncSpeed()
    if self.getIsEntered ~= nil and self:getIsEntered() then
        return self:getIsActiveForInput(true, false) and not self:getIsAIActive()
    end

    return false
end

function VehicleSpeedSync:vehicleSpeedSyncSetSyncedSpeedState(active, vehicleInRange, connectionLost, noEventSend)
    local spec = self.spec_vehicleSpeedSync

    if spec ~= nil then
        active = Utils.getNoNil(active and vehicleInRange ~= nil, false)

        if spec.speedSyncActive ~= active then
            if active and self:vehicleSpeedSyncGetVehicleIsInRange(vehicleInRange, spec.setRange) then
                spec.vehicleInRange = vehicleInRange
                spec.speedSyncActive = true
                spec.isWaiting = vehicleInRange:getLastSpeed() <= 0.5
            else
                spec.speedSyncActive = false
                spec.vehicleInRange = nil
                spec.isWaiting = false
            end

            if noEventSend == nil or not noEventSend then
                if not self.isServer then
                    g_client:getServerConnection():sendEvent(SetVehicleSpeedSyncEvent.new(self, spec.speedSyncActive, spec.vehicleInRange, connectionLost))
                else
                    local owner = self:getOwnerConnection()

                    if owner ~= nil then
                        owner:sendEvent(SetVehicleSpeedSyncEvent.new(self, spec.speedSyncActive, spec.vehicleInRange, connectionLost))
                    end
                end
            end

            if self.isClient and self.getIsEntered ~= nil and self:getIsEntered() then
                if spec.speedSyncActive then
                    g_soundManager:playSample(spec.samples.activate)

                    if spec.toggleSpeedSyncEventId ~= nil then
                        g_inputBinding:setActionEventText(spec.toggleSpeedSyncEventId, g_i18n:getText("action_deactivateSpeedSync", VehicleSpeedSync.MOD_NAME))
                    end
                else
                    if connectionLost == true then
                        spec.connectionLost = g_time + 2000
                        g_soundManager:playSample(spec.samples.connectionLost)
                    else
                        spec.connectionLost = nil
                        g_soundManager:playSample(spec.samples.deactivate)
                    end

                    if spec.toggleSpeedSyncEventId ~= nil then
                        g_inputBinding:setActionEventText(spec.toggleSpeedSyncEventId, g_i18n:getText("action_activateSpeedSync", VehicleSpeedSync.MOD_NAME))
                    end
                end
            end
        end
    end
end

function VehicleSpeedSync:vehicleSpeedSyncGetSyncedSpeedActive()
    local spec = self.spec_vehicleSpeedSync
    return spec ~= nil and spec.speedSyncActive and spec.vehicleInRange ~= nil
end

function VehicleSpeedSync:vehicleSpeedSyncGetSyncedSpeedWaiting()
    return self.spec_vehicleSpeedSync.isWaiting
end

function VehicleSpeedSync.getSyncedSpeedIconVisible()
    return VehicleSpeedSync.SPEED_SYNC_ACTIVE and VehicleSpeedSync.HUD_ICON_VISIBLE
end

function VehicleSpeedSync:vehicleSpeedSyncGetSyncedVehicleSpeed()
    if self:vehicleSpeedSyncGetSyncedSpeedActive() then
        local vehicleInRange = self.spec_vehicleSpeedSync.vehicleInRange
        local speed = vehicleInRange:getLastSpeed() * vehicleInRange.spec_motorized.speedDisplayScale

        if speed >= 0.5 then
            return math.floor(speed + 0.5)
        end
    end

    return 0
end

function VehicleSpeedSync:actionEventSpeedSyncState(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_vehicleSpeedSync

    if spec ~= nil then
        if not spec.speedSyncActive then
            local vehicleInRange = self:vehicleSpeedSyncFindVehicleInRange(spec.setRange)

            if vehicleInRange ~= nil then
                self:vehicleSpeedSyncSetSyncedSpeedState(true, vehicleInRange)
            else
                g_currentMission:showBlinkingWarning(spec.noVehicleInRangeText or "Not Possible!")
            end
        else
            self:vehicleSpeedSyncSetSyncedSpeedState(false)
        end
    end
end

function VehicleSpeedSync:actionEventSyncCruiseControl(actionName, inputValue, callbackState, isAnalog)
    local vehicleInRange

    if self:vehicleSpeedSyncGetSyncedSpeedActive() then
        vehicleInRange = self.spec_vehicleSpeedSync.vehicleInRange
    else
        vehicleInRange = self:vehicleSpeedSyncFindVehicleInRange(self.spec_vehicleSpeedSync.setRange)
    end

    if vehicleInRange ~= nil then
        local speed = math.floor(vehicleInRange:getLastSpeed() + 0.5)

        if speed > 0 then
            local drivableSpec = self.spec_drivable

            self:setCruiseControlMaxSpeed(speed)

            if speed ~= drivableSpec.cruiseControl.speedSent then
                if g_server ~= nil then
                    g_server:broadcastEvent(SetCruiseControlSpeedEvent:new(self, speed), nil, nil, self)
                else
                    g_client:getServerConnection():sendEvent(SetCruiseControlSpeedEvent:new(self, speed))
                end

                drivableSpec.cruiseControl.speedSent = speed
            end

            self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_ACTIVE)
        end
    else
        local spec = self.spec_vehicleSpeedSync

        if spec ~= nil then
            g_currentMission:showBlinkingWarning(spec.noVehicleInRangeText or "Not Possible!")
        else
            g_currentMission:showBlinkingWarning(g_i18n:getText("warning_actionNotAllowedNow"))
        end
    end
end

function VehicleSpeedSync:updateVehiclePhysics(superFunc, axisForward, axisSide, doHandbrake, dt)
    local spec = self.spec_vehicleSpeedSync

    if spec ~= nil and spec.speedSyncActive then
        if axisForward >= 0 then
            if self:vehicleSpeedSyncGetVehicleIsInRange(spec.vehicleInRange, spec.setRange) then
                local motor = self:getMotor()

                local speedLimit = motor:getSpeedLimit(true)
                local lastSpeed = spec.vehicleInRange:getLastSpeed()

                if lastSpeed > 0.5 and lastSpeed < speedLimit + 5 then
                    -- Allow acceleration without disconnecting.
                    if axisForward == 0 then
                        -- Use correct direction even with reverse driving.
						axisForward = math.sign((spec.vehicleInRange.movingDirection or 1) * spec.vehicleInRange:getReverserDirection())
                        motor:setSpeedLimit(math.min(lastSpeed, speedLimit))
                    end

                    spec.isWaiting = false
                elseif not spec.isWaiting then
                    self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, true)
                end
            else
                self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, true)
            end
        else
            self:vehicleSpeedSyncSetSyncedSpeedState(false)
        end
    end

    return superFunc(self, axisForward, axisSide, doHandbrake, dt)
end

function VehicleSpeedSync:getCanStartAIVehicle(superFunc)
    return superFunc(self) and not self:vehicleSpeedSyncGetSyncedSpeedActive()
end

function VehicleSpeedSync:setCruiseControlState(superFunc, state, noEventSend)
    if state ~= Drivable.CRUISECONTROL_STATE_OFF then
        self:vehicleSpeedSyncSetSyncedSpeedState(false, nil, false, true)
    end

    superFunc(self, state, noEventSend)
end
